/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true */
/*global define, require, _, $, module, describe, it, expect, afterEach, beforeEach */

"use strict";

var sinon = require("sinon"),
    DocinfoUtils = require("../DocinfoUtils");

describe("DocinfoUtils", function(){
    var sandbox = sinon.sandbox.create(),
        visitors = {
            accumulateLayerId: function (layer, context) {
                context.layersIds.push(layer.id);
                return true;
            },
            avoidChildren: function (layer, context) {
                context.layersIds.push(layer.id);
                return false;
            }
        },
        docinfo = {
            file: "my-doc.psd",
            layers: [
                { id: 1, artboard: {top:1} },
                { id: 2, artboard: {top:1}, layers: [
                    { id: 21 },
                    { id: 22 }
                ]},
                { id: 3, artboard: {top:1}, layers: [
                    { id: 31 },
                    { id: 32 }
                ]},
                { id: 4, artboard: {top:1}, layers: [
                    { id: 41 },
                ]},                
            ],
            _selectionById: [ 2, 21, 22, 3, 32, 41 ]
        };

    afterEach(function () {
        sandbox.restore();
    });

    describe("visitLayers", function () {
        it("visits layers in a depth-first way and accumulates visitor results", function () {
            // Visit all of the layers and accumulate their ids.
            sandbox.spy(visitors, "accumulateLayerId");
            var context = { layersIds: [] };

            DocinfoUtils.visitLayers(docinfo, visitors.accumulateLayerId, context);

            expect(visitors.accumulateLayerId.callCount).to.equal(9);
            expect(context.layersIds).to.eql([1, 2, 21, 22, 3, 31, 32, 4, 41]);
        });

        it("avoids visiting children when the visitor returns false", function () {
            sandbox.spy(visitors, "avoidChildren");
            var context = { layersIds: [] };

            DocinfoUtils.visitLayers(docinfo, visitors.avoidChildren, context);

            // Only the root's layer's are visited, but not their children.
            expect(visitors.avoidChildren.callCount).to.equal(4);
            expect(context.layersIds).to.eql([1, 2, 3, 4]);
        });
    });

    describe("getLayersForExport", function () {
        it("returns each selected layer except if the layer's siblings are all selected", function () {
            var layersForExport = DocinfoUtils.getLayersForExport(docinfo);
            expect(layersForExport).to.eql([
                docinfo.layers[1], // layer id 2
                docinfo.layers[2], // layer id 3
                docinfo.layers[2].layers[1], // layer id 32
                docinfo.layers[3].layers[0] // layer id 41
            ]);
        });

        it("returns a map of layer ids to parent artboards", function () {
            var parentArtboardMap = {},
                layersForExport = DocinfoUtils.getLayersForExport(docinfo, parentArtboardMap);

            // Check that we still get the layers for export correctly.
            expect(layersForExport).to.eql([
                docinfo.layers[1], // layer id 2
                docinfo.layers[2], // layer id 3
                docinfo.layers[2].layers[1], // layer id 32
                docinfo.layers[3].layers[0] // layer id 41
            ]);

            // Artboards shouldn't be parents of themselves.
            expect(parentArtboardMap[1]).to.eql(undefined);
            expect(parentArtboardMap[2]).to.eql(undefined);
            expect(parentArtboardMap[3]).to.eql(undefined);
            expect(parentArtboardMap[4]).to.eql(undefined);

            // These layers aren't part of the export because they are siblings of a completely selected parent.
            expect(parentArtboardMap[21]).to.eql(undefined);
            expect(parentArtboardMap[22]).to.eql(undefined);

            // Only the second layer here is actually selected.
            expect(parentArtboardMap[31]).to.eql(undefined);
            expect(parentArtboardMap[32].id).to.eql(3);

            expect(parentArtboardMap[41].id).to.eql(4);
        });
    });

    describe("getArtboards", function () {
        it("returns all artboards", function () {
            var artboards = DocinfoUtils.getArtboards(docinfo);
            expect(artboards).to.eql([
                docinfo.layers[0],
                docinfo.layers[1],
                docinfo.layers[2],
                docinfo.layers[3]
            ]);
        });
        it("isArtboard works correctly", function () {
            var layers = [
                    { id: 0, artboard: {top:1}},
                    { id: 1, artboard: {} },
                    { id: 2, artboard: true },
                    { id: 3, artboard: false },
                    { id: 4}
                ];
            expect(DocinfoUtils.layerIsArtboard(layers[0])).to.be.true;
            expect(DocinfoUtils.layerIsArtboard(layers[1])).to.be.false;
            expect(DocinfoUtils.layerIsArtboard(layers[2])).to.be.false;
            expect(DocinfoUtils.layerIsArtboard(layers[3])).to.be.false;
            expect(DocinfoUtils.layerIsArtboard(layers[4])).to.be.false;
        });
    });
});
